#region Fileheader
//
// $Id: AxisTamaProgram.cs 8561 2008-08-14 09:48:55Z chm $
// 
// Copyright  2007 Triamec AG
// 
#endregion Fileheader

#region Notes
//  LK, 9. March 2010
//
//  Tama Notes
//  ==========
//  
//  The task entry points are defined by the keyword "TamaTask"
//  and are immediately followed by the desired function.
//  The "TamaTask" parameter specifies the type of task with the following
//  properties:
//  
//  [TamaTask(Task.AsynchronousMain)] : Task executed as fast as possible
//                                      -> useful for complex state machines
//                                      -> and time uncritical operations
//  [TamaTask(Task.IsochronousMain)]  : Task executed every 100us
//                                      -> useful for hard realtime operations (e.g. kinematics)
//                                      -> caution: very limited computing resources
//  [TamaTask(Task.Axis1Coupling)]    : Task executed every 100us (1 axis system)
//                                      -> axis 1 coupling calculation before entering the path interpolation
//  [TamaTask(Task.Axis1Init)]        : Task executed once when setting the path planner to synchronous motion
//                                      -> some preparing functions before entering synchronous motion of axis 1
//  [TamaTask(Task.Axis2Coupling)]    : Task executed every 200us (2 axis systems)
//                                      -> axis 2 coupling calculation before entering the path interpolation
//  [TamaTask(Task.Axis2Init)]        : Task executed once when setting the path planner to synchronous motion
//                                      -> some preparing functions before entering synchronous motion of axis 2
//
//   Static variables defined with "static ..." are persistent (e.g. static int counter = 0). This means that the values will be stored
//   when the task calculation has completed. They can be initialized before entering the task for the first time.
//   The static variable initialization takes place when downloading the code. A start/stop of the task does not
//   initialize the variables.
//
//   Since every task has its own stack, it is not possible to exchange information from task to task by using static variables.
//   Use the Tama general purpose registers instead.
//
//   How to find a register?
//   -> type an "R" and fellow the Intellisense aid of the visual studio.
//   The registers are organized like in the explorer register tree.
//
#endregion Notes

using Triamec.TriaLink;
using Triamec.Tama.Vmid5;
using Triamec.Tama.Rlid5;

namespace TSX50_RLID5 {
	/// <summary>
	/// Tama program for the
	/// axis of the module catalog.
	/// Uses Register.Tama.Variables.GenPurposeIntVar0 for time delay counting;
	/// and  Register.Tama.Variables.GenPurposeIntVar4 for the code id
	/// </summary>

	[Tama]
	static class TamaTwinCatEndswitches {

		// input signals:

		// In4 = din(3) = position limit in negative direction reached
		// In5 = din(4) = position limit in positive direction reached
		// moves stop immediately when reaching the bad zone
		// another move starting within the bad zone is stopped when prolonging longer than cDelayTicks

		// the number of 0.1ms ticks until a bad move stops when already in the bad zone
		// if used with Beckhoff, this must be longer than the MAIN_FAST time.
		const int cDelayTicks = 15;
		// the code id to be stored in GenPurposeIntVar4
		const int cCodeId = 712;

		#region user functions
		/// <summary>
		/// User Subroutines 
		/// </summary>
		///

		private static bool GetDigitalIn(int channel) {
			return ((Register.General.Signals.InputBits & (1 << channel)) != 0);
		}

		#endregion user functions




		#region isochronous Tama main application

		/// <summary>
		/// Stops the drive depending on two end switches.
		/// </summary>
		[TamaTask(Task.IsochronousMain)]
		static void IsochronApplication() {
			// active low inputs
			bool lowLimit = !GetDigitalIn(3);
			bool highLimit = !GetDigitalIn(4);

			AxisState axisState = Register.Axes_0.Signals.General.AxisState;
			if (axisState == AxisState.ErrorStopping) return;

			switch (Register.Tama.IsochronousMainState) {
				case 0:
					// reset delay counter
					Register.Tama.Variables.GenPurposeIntVar4 = cCodeId;
					Register.Tama.Variables.GenPurposeIntVar0 = 0;

					// test for end-switches and change to state 1=highLimit or 2=lowLimit
					// Issue a full dynamics stop when lower limit is reached and moving in negative direction, or
					// when higher limit is reached and moving in positive direction.
					if (highLimit) {

						Register.Tama.IsochronousMainState = 1;
						if ((Register.Axes_0.Signals.PathPlanner.Velocity > 0f) && (axisState > AxisState.Standstill)) {
							Register.Axes_0.Commands.PathPlanner.Command = PathPlannerCommand.EmergencyStop;
						}
					}
					if (lowLimit) {

						Register.Tama.IsochronousMainState = 2;
						if ((Register.Axes_0.Signals.PathPlanner.Velocity < 0f) && (axisState > AxisState.Standstill)) {
							Register.Axes_0.Commands.PathPlanner.Command = PathPlannerCommand.EmergencyStop;
						}
					}
					break;

				case 1:
					// go back to idle state if limit switch got released
					if (!highLimit) {
						Register.Tama.IsochronousMainState = 0;
						return;
					}
					// we are still in the limit. If there is a new move command in the
					// wrong direction for some time, then stop again. 
					if ((Register.Axes_0.Signals.PathPlanner.Velocity > 0f) && (axisState > AxisState.Standstill)) {
						if (Register.Tama.Variables.GenPurposeIntVar0 > cDelayTicks) {
							Register.Axes_0.Commands.PathPlanner.Command = PathPlannerCommand.EmergencyStop;
						} else {
							Register.Tama.Variables.GenPurposeIntVar0 = Register.Tama.Variables.GenPurposeIntVar0 + 1;
						}
					} else {
						Register.Tama.Variables.GenPurposeIntVar0 = 0;
					}
					break;

				case 2:
					// go back to idle state if limit switch got released
					if (!lowLimit) {
						Register.Tama.IsochronousMainState = 0;
						return;
					}
					// we are still in the limit. If there is a new move command in the
					// wrong direction for some time, then stop again. 
					if ((Register.Axes_0.Signals.PathPlanner.Velocity < 0f) && (axisState > AxisState.Standstill)) {
						if (Register.Tama.Variables.GenPurposeIntVar0 > cDelayTicks) {
							Register.Axes_0.Commands.PathPlanner.Command = PathPlannerCommand.EmergencyStop;
						} else {
							Register.Tama.Variables.GenPurposeIntVar0 = Register.Tama.Variables.GenPurposeIntVar0 + 1;
						}
					} else {
						Register.Tama.Variables.GenPurposeIntVar0 = 0;
					}
					break;

				default:
					Register.Tama.IsochronousMainState = 0;
					break;

			}

		}
		#endregion isochronous Tama Main application

	}
}